﻿using System.Collections.Generic;
using System.Linq;
using Microsoft.Data.SqlClient;
using SoftPi.Tariscope;
using SoftPi.Tariscope.Common;
using SoftPi.Tariscope.DAL;
using SoftPI.Tariscope.WebAdministration.Observer.Scripting.Interfaces;
using SoftPI.Tariscope.WebAdministration.Observer.Scripting.Models;
using System.Globalization;
using System.IO;
using System;
using Newtonsoft.Json;

/// <summary>
/// Is used to inform a user that n% of limit is spended. 
/// </summary>
class CriticalPercentageLimitScript : IScript
{
    // Change this constants to set your specific parameters
    public int[] chargePercentage = new int[] { 80, 95 };                       //define percentages for which notifications are needed    
    const string emailAddress = "support@softpiua.com";                         //email address for receiving notifications
    const string emailSubject = "{0:F2}% of limit is spended.";                 //email subject
    const string emailBody = "Department {0} reached limit of {1:F2}%";         //email body

    const string configPath = @"C:\ProgramData\Tariscope\Configs\Observers\{0}.json";    //default path to Observers' configs
    private IScriptHost _host;

    public void Init(IScriptHost host)
    {
        _host = host;
    }

    public void Main(object Parameters)
    {
        NewCallActionParameters callParams = (NewCallActionParameters)Parameters;

        _host.AddEvent("Script critical-percentage-limit-department started");

        var departmentInfo = LoadDepartment(callParams.Id);

        if (departmentInfo.Item1 == 0
         || departmentInfo.Item2 == 0)
            return;

        var actualPercentage = LoadPercentage((departmentInfo.Item1, departmentInfo.Item2));
        var nextAutoConnectDate = LoadAutoConnectDate().Date;
        _host.AddEvent($"next autoconnect date: {nextAutoConnectDate}");

        using (var cn = new SqlConnection(_host.DatabaseConnectionString))
        {
            cn.Open();

            for (int i = 0; i < chargePercentage.Length; i++)
                if (actualPercentage >= chargePercentage[i])
                {
                    _host.AddEvent($"{chargePercentage[i]}% mark reached.");

                    var lastMailDate = LoadLastMailDate(chargePercentage[i]);

                    if (nextAutoConnectDate > lastMailDate)
                    {
                        _host.AddEvent("Need to send mail");
                        SendMail(chargePercentage[i], departmentInfo.Item3);
                        SaveLastMailDate(chargePercentage[i], nextAutoConnectDate);
                        break;
                    }
                }
        }

        _host.AddEvent("Script critical-percentage-limit-department finished");
    }

    public void SendMail(int percentage, string departmentName)
    {
        using (var cn = new SqlConnection(_host.DatabaseConnectionString))
        {
            cn.Open();
            var subject = string.Format(emailSubject, percentage);
            var message = string.Format(emailBody, departmentName, percentage);
            _host.SendMail(emailAddress, subject, message);
        }
    }

    public (int, int, string) LoadDepartment(long callId)
    {
        int nodeId, departmentId;
        string departmentName = string.Empty;

        var cmdText = @"select 
                            ISNULL(C.NodeID, 0), 
                            ISNULL(FromDepartmentID, 0), 
                            D.Name 
                        from 
                            viCalls C 
                       join Departments D 
                       on C.FromDepartmentID = D.ID 
                       where 
                            C.ID = @callId";

        using (var cn = new SqlConnection(_host.DatabaseConnectionString))
        {
            cn.Open();
            var cmd = MSSQL.NewQuery(cn, cmdText);
            cmd.Parameters.AddWithValue("@callId", callId);
            var reader = cmd.ExecuteReader();

            reader.Read();
            nodeId = reader.GetInt32(0);
            departmentId = reader.GetInt32(1);
            departmentName = reader.GetString(2);
            reader.Close();

            return (nodeId, departmentId, departmentName);
        }
    }

    public decimal LoadPercentage((int, int) departmentInfo)
    {
        var cmdText = @"SELECT
		                    ISNULL(DC.Debit, 0) / ISNULL(DC.Credit, 0) * 100
	                     FROM 
		                    dbo.Departments AS D INNER JOIN 
		                    dbo.DepartmentCredits AS DC ON DC.DepartmentID=D.ID LEFT JOIN 
		                    dbo.Departments AS DParents ON D.ParentID=DParents.ID LEFT JOIN
		                    dbo.Categories as C ON DC.CategoryID=C.ID
	                     WHERE
                             D.NodeId = @nodeId 
                         AND D.ID = @departmentId";

        decimal percentage = 0;

        using (var cn = new SqlConnection(_host.DatabaseConnectionString))
        {
            cn.Open();
            var cmd = MSSQL.NewQuery(cn, cmdText);
            cmd.Parameters.AddWithValue("@nodeId", departmentInfo.Item1);
            cmd.Parameters.AddWithValue("@departmentId", departmentInfo.Item2);
            percentage = (decimal)cmd.ExecuteScalar();
        }

        return percentage;
    }

    public string LoadLastMailDates()
    {
        var cmdText = @"SELECT
		                    [Value]
	                     FROM 
		                    dbo.Settings
	                     WHERE
                             [Key] = 'RestrictNotifyDates'";

        string datesStr = string.Empty;

        using (var cn = new SqlConnection(_host.DatabaseConnectionString))
        {
            cn.Open();
            var cmd = MSSQL.NewQuery(cn, cmdText);
            datesStr = cmd.ExecuteScalar().ToString();
        }

        return datesStr;
    }

    public DateTime LoadLastMailDate(int percentage)
    {
        string datesStr = LoadLastMailDates();

        var percentTemplate = percentage.ToString() + '=';
        string lastDateStr = string.Empty;

        foreach (var date in datesStr.Split(';', StringSplitOptions.RemoveEmptyEntries))
        {
            if (date.StartsWith(percentTemplate))
            {
                lastDateStr = date.Split('=', StringSplitOptions.RemoveEmptyEntries).Last();
                break;
            }
        }

        return lastDateStr == string.Empty
            ? new DateTime()
            : DateTime.ParseExact(lastDateStr, "yyyyMMdd", CultureInfo.InvariantCulture);
    }

    public void SaveLastMailDate(int percentage, DateTime date)
    {
        var percentTemplate = percentage.ToString() + '=';

        var datesStr = LoadLastMailDates();
        var datePairs = datesStr.Split(';', StringSplitOptions.RemoveEmptyEntries).ToList();

        var chosenPersentDate = datePairs.FirstOrDefault(x => x.StartsWith(percentTemplate));

        if (chosenPersentDate != null)
            datesStr = datesStr.Replace(chosenPersentDate, percentTemplate + date.ToString("yyyyMMdd"));
        else
            datesStr = datesStr + ';' + percentTemplate + date.ToString("yyyyMMdd");

        var cmdText = "Update dbo.Settings SET Value = @newDate WHERE [Key] = 'RestrictNotifyDates'";

        using (var cn = new SqlConnection(_host.DatabaseConnectionString))
        {
            cn.Open();
            var cmd = MSSQL.NewQuery(cn, cmdText);
            cmd.Parameters.AddWithValue("@newDate", datesStr);
            cmd.ExecuteNonQuery();
        }
    }

    public DateTime LoadAutoConnectDate()
    {
        var path = string.Format(configPath, _host.ProfileName);

        string configStr = string.Empty;

        try
        {
            configStr = File.ReadAllText(path);
        }
        catch (Exception e)
        {
            var nextMonth = DateTime.Now.AddMonths(1);
            return new DateTime(nextMonth.Year, nextMonth.Month, 1);
        }

        return JsonConvert.DeserializeObject<ObserverConfiguration>(configStr).NextSubscribersAutoConnect;
    }
}
